;           RTC & CTC demo program for MicroProfessor MPF-I(B)

; Based on Zilog Z80 Family Data Book, January 1989, p 366
; by fjkraan@electrickery 2022-01-28
; z80asm version (https://github.com/udo-munk/z80pack/tree/master/z80asm)
; This program assumes a Z80 CTC at port 40h and the standard MPF-I(B) 
; hex display.

; MPF-I CPU clock: 1.79 MHz. A prescaler of 256 and a time constant of 84
; and a division of again 83 in the ISR results in a second update 
; interval of approx. 1 s. A real RTC chip could supply long term
; accuracy.

; CTCCLOCK version 0.2, 2022-02-01

; Usage: enter the current time HH:MM:SS at MANBUF and run SETCLK.
;        This will set the time values and jump to BEGIN to display and 
;        update the clock. 

;           Memory organisation
RAM:    EQU     1980h
ROM:    EQU     2200h
RAMSIZ: EQU     90h

;           CTC ports
CTCBASE:EQU     40h
CTC0:   EQU     CTCBASE + 0         ;CTC 0 PORT
CTC1:   EQU     CTCBASE + 1         ;CTC 1 PORT
CTC2:   EQU     CTCBASE + 2         ;CTC 2 PORT
CTC3:   EQU     CTCBASE + 3         ;CTC 3 PORT
;LITES:   EQU     0C0h                ;LIGHT PORT

;           CTC Constants
CCW:    EQU     1
INTEN:  EQU     80h     ; Interrupt enable
CTRMODE:EQU     40h     ; Counter mode
P256:   EQU     20h     ; Prescaler 256
RISEDO: EQU     10h     ; Rising Edge Trigger
PSTRT:  EQU     8       ; Trigger by CLK/TRG
TCLOAD: EQU     4       ; Time constant in following byte load (TIME)
RESET:  EQU     2       ; Software Reset
TIMER:  EQU     84      ; CTC TIMER VALUE 
COUNTER:EQU     83      ; ISR COUNT1 value

; Monitor ROM routines
SCAN:   EQU     05FEh       ; 
SCAN1:  EQU     0624h       ; input: (IX), output: carry & A
HEX7SEG:EQU     0678h       ; input: A (2 * BCD), output (HL). HL+2

;       RAM organisation
        ORG     RAM
MANBUF:                 ; buffer to store BCD values when manually 
                        ;  setting time and date
        DEFB    00      ; seconds MANBUF+0
        DEFB    00      ; minutes MANBUF+1
        DEFB    00      ; hours   MANBUF+2
        DEFB    00      ; day-of-week
        DEFB    00      ; days    MANBUF+4
        DEFB    00      ; months  MANBUF+5
        DEFB    00      ; years   MANBUF+6
        
RTCBUF: ; data here stored as binary
        DEFB    00      ; seconds RTCBUF+0
        DEFB    00      ; minutes RTCBUF+1
        DEFB    00      ; hours   RTCBUF+2
        DEFB    00      ; day-of-week
        DEFB    00      ; days    RTCBUF+4
        DEFB    00      ; months  RTCBUF+5
        DEFB    00      ; years   RTCBUF+6
    
DISPBUF:                ; six digits, from right to left, to be filled 
                        ;  with 7-segment display patterns
        DEFW    0000    ; seconds / years
        DEFW    0000    ; minutes / months
        DEFW    0000    ; hours   / days

        DEFS    64      ; STACK AREA
STAK:   EQU     $
COUNT:  DEFB    00      ; ICT COUNT VALUE
SECFLAG:DEFB    00      ; second boundary flag, triggers RTCBUF update 
MROFFS: EQU     RTCBUF - MANBUF

RB_SC:  EQU     RTCBUF + 0
RB_MI:  EQU     RTCBUF + 1
RB_HR:  EQU     RTCBUF + 2

;;          *** MAIN PROGRAM ***

        ORG     ROM
        JP      BEGIN
        
        JP      SETCLK
        
        ORG     $ & 0FFF0h | 10h
INTVEC: 
        DEFW    ICTC0
        DEFW    ICTC1
        DEFW    ICTC2
        DEFW    ICTC3
        
BEGIN:  
        LD      SP,STAK         ; INIT SP
        IM      2               ; VECTOR INTERRUPT MODE
        LD      A,INTVEC / 256  ; UPPER VECTOR BYTE
        LD      I,A
        CALL    INIT            ; INIT DEVICES (CTC & memory)
        EI                      ; ALLOW INTERRUPTS

LOOP:        
        LD      A, (SECFLAG)
        CP      1
        LD      A, 0            ; clear secflag, regardless of state
        LD      (SECFLAG), A
        JR      NZ, NO_ACT      ; 
        CALL    UPDTIME
NO_ACT:
        CALL    DSP             ; Show counters on display
        JR      LOOP            ; LOOP FOREVER
        
INIT:
        LD      A, INTEN + P256 + TCLOAD + RESET + CCW
        OUT     (CTC0), A       ; SET CTC MODE
        LD      A, TIMER
        OUT     (CTC0), A       ; SET TIME CONSTANT
        LD      A, INTVEC & 11111110b
        OUT     (CTC0), A       ; SET VECTOR VALUE
        LD      A, COUNTER      ; INIT COUNTER VALUE
        LD      (COUNT), A
        XOR      A
        LD      (SECFLAG), A
        
        LD      (DISPBUF), A     ; clear display buffer
        LD      (DISPBUF + 1), A
        LD      (DISPBUF + 2), A
        LD      (DISPBUF + 3), A
        LD      (DISPBUF + 4), A
        LD      (DISPBUF + 5), A
        
        RET
        
UPDTIME:
        ; update seconds up to 60
        ; update minutes up to 60 when seconds == 60, reset seconds
        ; update hours up to 24 when minutes == 60, reset minutes
        ; update date when hours == 24, reset hours
        ; ....
        LD      A, (RB_SC)
        INC     A
        CP      60
        JR      Z, MIN_UPD
        LD      (RB_SC), A
        JR      UPDDONE
        
MIN_UPD:
        LD      A, 0
        LD      (RB_SC), A          ; set seconds to zero
        LD      A, (RB_MI)
        INC     A                   ; update minutes
        CP      60
        JR      Z, HR_UPD
        LD      (RB_MI), A
        JR      UPDDONE
        
HR_UPD:
        LD      A, 0
        LD      (RB_MI), A          ; set minutes to zero
        LD      A, (RB_HR)
        INC     A                   ; update hours
        CP      24
        JR      Z, DT_UPD
        LD      (RB_HR), A
        JR      UPDDONE
        
DT_UPD:     ; date update not implemented yet; 
        LD      A, 0
        LD      (RB_HR), A          ; set hours to zero

UPDDONE:
        RET

DSP:
        LD      HL, DISPBUF     ; point to rightmost value
        LD      BC, RTCBUF
        LD      A, (BC)
        CALL    BIN2BCD
        CALL    HEX7SEG
        INC     BC              ; points to minutes value
        LD      A, (BC)
        CALL    BIN2BCD
        CALL    HEX7SEG
        INC     BC              ; points to hours value
        LD      A, (BC)
        CALL    BIN2BCD
        CALL    HEX7SEG
       
        LD      IX, DISPBUF
        CALL    SCAN1           ; call ROM display routine
        
        RET
        
BCD2BIN:
        ; input; A contains 2 BCD digits - MSN and LSN
        ; output; A contains a binary value
        PUSH    BC
        PUSH    AF
        AND     0F0h    ; create MSN value only
        RRCA
        LD      C, A    ; keep MSN * 8 
        RRCA
        RRCA
        ADD     A, C    ; add MSN * 2 to MSN * 8
        LD      C, A    ; keep the MSN * 10
        POP     AF
        AND     00Fh    ; create LSN value only
        ADD     A, C       ; create complete binary value
        POP     BC
        RET
        
BIN2BCD:
        ; input; A contains a binary value less than 100 (MSNb & LSNb)
        ; output; A contains two BCD digits - MSN and LSN
        PUSH    BC
        LD      B, 0            ; start with zero MSB
BI2BLP  SUB     10
        JP      C, BI2BRY       ; done counting 10s
        INC     B
        JR      BI2BLP          ; continue until A < 10
BI2BRY  ADD     A, 10           ; compensate 
        RLC     B               ; move 10s to upper nibble
        RLC     B
        RLC     B
        RLC     B
        ADD     A, B            ; merge nibbles
        POP     BC

        RET
        
        ; Converts a BCD value into its binary equivalent as this is 
        ;  easier to update
SETCLK:
        LD      IX, MANBUF          ; point to RTCBUF seconds
        
        LD      A, (IX + 0)         ; load from MANBUF seconds
        CALL    BCD2BIN
        LD      (IX + MROFFS), A    ; store to RTCBUF seconds
        INC     IX
        
        LD      A, (IX + 0)         ; minutes
        CALL    BCD2BIN
        LD      (IX + MROFFS), A    ; 
        INC     IX
        
        LD      A, (IX + 0)         ; hours
        CALL    BCD2BIN
        LD      (IX + MROFFS), A    ; 
        INC     IX

        LD      A, (IX + 0)         ; day of week
        CALL    BCD2BIN
        LD      (IX + MROFFS), A    ; 
        INC     IX

        LD      A, (IX + 0)         ; day of month
        CALL    BCD2BIN
        LD      (IX + MROFFS), A    ; 
        INC     IX

        LD      A, (IX + 0)         ; months
        CALL    BCD2BIN
        LD      (IX + MROFFS), A    ; 
        INC     IX
       
        LD      A, (IX + 0)         ; years
        CALL    BCD2BIN
        LD      (IX + MROFFS), A    ; 
        
        JP      BEGIN           ; start clock
        RST     0
        
        
;       INTERRUPT SERVICE ROUTINES 

ICTC1:
ICTC2:
ICTC3:
        EI
        RETI                    ; DUMMY ROUTINES
        
ICTC0:
        PUSH    AF
        LD      A, (COUNT)      ; CHANGE TIMER COUNT
        DEC     A
        LD      (COUNT), A
;        OUT     (LITES), A
        CP      0
        JR      NZ, IC0DONE     ; (COUNT) not yet zero, exit ISR
        LD      A, COUNTER      ;  ELSE, RESET COUNTER VALUE
        LD      (COUNT), A
        
        LD      A, 1
        LD      (SECFLAG), A
        
IC0DONE:
        POP     AF
        EI
        RETI

        END
