;           CTC TEST PROGRAM for MicroProfessor MPF-I(B)

;           THIS PROGRAM USES THE CTC IN CONTINUOUS
;           TIMER MODE. THE CTC COUNTS SYSTEM  CLOCK
;           PULSES AND INTERRUPTS EVERY 84 PULSES,
;           THEN DECREMENTS A COUNT, SHOWS TIMER AND
;           AND COUNTER VALUES ON THE HEX DISPLAY.

; Based on Zilog Z80 Family Data Book, January 1989, p 366
; by fjkraan@electrickery 2022-01-28
; z80asm version (https://github.com/udo-munk/z80pack/tree/master/z80asm)
; This program assumes a CTC at port 40h and the standard MPF-I(B) hex
; display.

; MPF-I CPU clock: 1.79 MHz. A prescaler of 256 and a time constant of 84h
; results in a pulse interval of approx. 1 s pulse at ZC/TO0 (pin 7).


;           PROGRAM EQUATES

CTCBASE: EQU    40h
CTC0:   EQU     CTCBASE + 0         ;CTC 0 PORT
CTC1:   EQU     CTCBASE + 1         ;CTC 1 PORT
CTC2:   EQU     CTCBASE + 2         ;CTC 2 PORT
CTC3:   EQU     CTCBASE + 3         ;CTC 3 PORT
;LITE:   EQU     0C0h                ;LIGHT PORT
RAM:    EQU     1980h               ;RAM START ADDRESS
ROM:    EQU     2200h
RAMSIZ: EQU     90h
TIME:   EQU     84                 ;COUNT VALUE 

; Monitor ROM routines
SCAN1:  EQU     0624h
HEX7SEG:    EQU 0678h

;           CTC EQUATES
CCW:    EQU     1
INTEN:  EQU     80h     ; Interrupt enable
CTRMODE:  EQU   40h     ; Counter mode
P256:   EQU     20h     ; Prescaler 256
RISEDO: EQU     10h     ; Rising Edge Trigger
PSTRT:  EQU     8       ; Trigger by CLK/TRG
TCLOAD: EQU     4       ; Time constant in following byte load (TIME)
RESET:  EQU     2       ; Software Reset

;       RAM organisation
        ORG     RAM
        DEFS    64              ;STACK AREA
STAK:   EQU     $
COUNT:  DEFS    1               ;TIMER COUNT VALUE
DISPLSB: DEFS   1               ;LSB DISPLAY BYTE
DISPMSB: DEFS   1               ;MSB DISPLAY BYTE
DSPBUF: 
        DEFS    6               ; Hex display buffer, right to left


;;          *** MAIN PROGRAM ***

        ORG     ROM
        JP      BEGIN

        ORG     $ & 0FFF0h | 10h
INTVEC: 
        DEFW    ICTC0
        DEFW    ICTC1
        DEFW    ICTC2
        DEFW    ICTC3
        
BEGIN:  
        LD      SP,STAK         ;INIT SP
        IM      2               ;VECTOR INTERRUPT MODE
        LD      A,INTVEC / 256  ;UPPER VECTOR BYTE
        LD      I,A
        CALL    INIT            ;INIT DEVICES (CTC & memory)
        EI                      ;ALLOW INTERRUPTS

LOOP:        
        CALL    DSP             ; Show counters on display
        JR      LOOP            ;LOOP FOREVER
        
DSP:
        LD      A, (COUNT)      ; CTC0 timer count on right two digits
        LD      HL, DSPBUF
        CALL    HEX7SEG
        LD      A, (DISPLSB)    ; overflow LSB count on middle two displays
        LD      HL, DSPBUF + 2
        CALL    HEX7SEG
        LD      A, (DISPMSB)    ; overflow MSB count on left two displays
        LD      HL, DSPBUF + 4
        CALL    HEX7SEG
        
        LD      IX, DSPBUF
        CALL    SCAN1           ; call ROM display routine
        
        RET
        
INIT:
        LD      A, INTEN + P256 + TCLOAD + RESET + CCW
        OUT     (CTC0),A        ;SET CTC MODE
        LD      A, TIME
        OUT     (CTC0), A       ;SET TIME CONSTANT
        LD      A,INTVEC & 11111000b
        OUT     (CTC0), A       ;SET VECTOR VALUE
        XOR     A
        LD      (DISPLSB), A    ;CLEAR DISPLAY LS BYTE
        LD      (DISPMSB), A    ;CLEAR DISPLAY MS BYTE
        LD      A,TIME          ;INIT TIMER VALUE
        LD      (COUNT), A
        
        XOR      A
        LD      (DSPBUF), A     ; clear display buffer
        LD      (DSPBUF + 1), A
        LD      (DSPBUF + 2), A
        LD      (DSPBUF + 3), A
        LD      (DSPBUF + 4), A
        LD      (DSPBUF + 5), A
        LD      (DISPLSB), A
        LD      (DISPMSB), A
        
        RET

;       INTERRUPT SERVICE ROUTINES 

ICTC1:
ICTC2:
ICTC3:
        EI
        RETI                    ;DUMMY ROUTINES
        
ICTC0:
        PUSH    AF
        LD      A,(COUNT)       ;CHANGE TIMER COUNT
        DEC     A
        LD      (COUNT), A
        JP      NZ, IC0DONE
        LD      A, TIME          ;ELSE, REST TIMER VALUE
        LD      (COUNT), A
        LD      A, (DISPLSB)     ;update overflow LSB
        INC     A
        LD      (DISPLSB),A
        JP      NZ, IC0DONE
        LD      A, (DISPMSB)     ;update overflow MSB
        INC     A
        LD      (DISPMSB), A
IC0DONE:
        POP     AF
        EI
        RETI

        END
