/*
 * dfi2dmk: Convert floppy disk image to .dmk format.
 * Adapted from cw2dmk
 * Copyright (C) 2000 Timothy Mann
 * Id: cw2dmk.c,v 1.38 2010/01/15 20:32:56 mann Exp 
 * 
 * The extra code bits copyright (C) 2012, 2013 Fred Jan Kraan
 * dfi2dmk.c, v 0.1 2013/01/04 
 *
 * Depends on Linux Catweasel driver code by Michael Krause
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*#define DEBUG5 1*/ /* Extra checking to detect Catweasel MK1 memory errors */
#define DEBUG5_BYTE 0x7e

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <string.h>
#include <math.h>
#include <signal.h>
#if linux
#include <sys/io.h>
#endif
#include "crc.c"
#include "cwfloppy.h"
#include "dmk.h"
#include "kind.h"
#include "version.h"
#include "decoder.h"
#include "dfi2dmk.h"

// clean_up, do_histogram, main
//struct catweasel_contr c;
// dmk_data, dmk_idam, dmk_iam, dmk_write, dmk_write_header, dmk_init_track, dmk_check_wrap_around, main
dmk_header_t dmk_header;

// sec_size, usage, main
int maxsize = 3;  /* 177x/179x look at only low-order 2 bits */


char* plu(val)
{
  return (val == 1) ? "" : "s";
}

// usage, main
int out_level = OUT_TSUMMARY;
// main
int out_file_level = OUT_QUIET;
char *out_file_name;
FILE* out_file;

/* Log a message. */
void msg(int level, const char *fmt, ...)
{
  va_list args;

  if (level <= out_level &&
      !(level == OUT_RAW && out_level != OUT_RAW) &&
      !(level == OUT_HEX && out_level == OUT_RAW)) {
    va_start(args, fmt);
    vfprintf(stdout, fmt, args);
    va_end(args);
  }
  if (out_file && level <= out_file_level &&
      !(level == OUT_RAW && out_file_level != OUT_RAW) &&
      !(level == OUT_HEX && out_file_level == OUT_RAW)) {
    va_start(args, fmt);
    vfprintf(out_file, fmt, args);
    va_end(args);
  }
}

/* True if we are ignoring data while waiting for an iam or for the
   first idam */
int dmk_awaiting_track_start(DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  if (dmkParams->dmk_iam_pos == -1) {
    return !decoderParams->hole && (unsigned char*) dmkParams->dmk_idam_p == dmkParams->dmk_track;
  } else {
    return dmkParams->dmk_awaiting_iam;
  }
}

int dmk_in_range(DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  if (dmkParams->dmk_full) return 0;
  if (dmkParams->dmk_ignored_count < dmkParams->dmk_ignore) {
    dmkParams->dmk_ignored_count++;
    return 0;
  }
  /* Stop at leading edge of last index unless in sector data. */
  if (decoderParams->hole && decoderParams->index_edge >= 3 && decoderParams->dbyte == -1) {
    msg(OUT_HEX, "[index edge %d] ", decoderParams->index_edge);
    dmkParams->dmk_full = 1;
    return 0;
  }
  return 1;
}

void dmk_data(unsigned char byte, int encoding, DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  if (!dmk_in_range(decoderParams, dmkParams)) return;
  if (dmkParams->dmk_awaiting_dam && byte >= 0xf8 && byte <= 0xfd) {
    /* Kludge: DMK doesn't tag DAMs, so noise after the ID bytes
       but before the DAM must not have the data bit pattern of a DAM */
    byte = 0xf0;
  }
  if (dmkParams->dmk_data_p - dmkParams->dmk_track <= dmk_header.tracklen - 2) {
    *dmkParams->dmk_data_p++ = byte;
    if (encoding == FM && !(dmk_header.options & DMK_SDEN_OPT)) {
      *dmkParams->dmk_data_p++ = byte;
    }
  }
  if (dmkParams->dmk_data_p - dmkParams->dmk_track > dmk_header.tracklen - 2) {
    /* No room for more bytes after this one */
    msg(OUT_HEX, "[DMK track buffer full] ");
    dmkParams->dmk_full = 1;
  }
}

void dmk_idam(unsigned char byte, int encoding, DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  unsigned short idamp;
  if (!dmk_in_range(decoderParams, dmkParams)) return;

  if (!dmkParams->dmk_awaiting_iam && dmk_awaiting_track_start(decoderParams, dmkParams)) {
    /* In this mode, we position the first IDAM a nominal distance
       from the start of the track, to make sure that (1) the whole
       track will fit and (2) if dmk2cw is used to write the image
       back to a real disk, the first IDAM won't be too close to the
       index hole.  */
#define GAP1PLUS 48
    int bytesread = dmkParams->dmk_data_p - (dmkParams->dmk_track + DMK_TKHDR_SIZE);
    if (bytesread < GAP1PLUS) {
      /* Not enough bytes read yet.  Move read bytes forward and add fill. */
      memmove(dmkParams->dmk_track + DMK_TKHDR_SIZE + GAP1PLUS - bytesread,
	      dmkParams->dmk_track + DMK_TKHDR_SIZE,
	      bytesread);
      memset(dmkParams->dmk_track + DMK_TKHDR_SIZE,
	     (encoding == MFM) ? 0x4e : 0xff,
	     GAP1PLUS - bytesread);
    } else {
      /* Too many bytes read.  Move last GAP1PLUS back and throw rest away. */
      memmove(dmkParams->dmk_track + DMK_TKHDR_SIZE,
	      dmkParams->dmk_track + DMK_TKHDR_SIZE + bytesread - GAP1PLUS,
	      GAP1PLUS);
    }
    dmkParams->dmk_data_p = dmkParams->dmk_track + DMK_TKHDR_SIZE + GAP1PLUS;
  }

  dmkParams->dmk_awaiting_dam = 0;
  dmkParams->dmk_valid_id = 0;
  idamp = dmkParams->dmk_data_p - dmkParams->dmk_track;
  if (encoding == MFM) {
    idamp |= DMK_DDEN_FLAG;
  }
  if (dmkParams->dmk_data_p < dmkParams->dmk_track + dmk_header.tracklen) {
    if ((unsigned char*) dmkParams->dmk_idam_p >= dmkParams->dmk_track + DMK_TKHDR_SIZE) {
      msg(OUT_ERRORS, "[too many AMs on track] ");
      decoderParams->errcount++;
    } else {
      *dmkParams->dmk_idam_p++ = idamp;
      decoderParams->ibyte = 0;
      dmk_data(byte, encoding, decoderParams, dmkParams);
    }
  }
}

void dmk_iam(unsigned char byte, int encoding, DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  if (!dmk_in_range(decoderParams, dmkParams)) return;

  if (dmkParams->dmk_iam_pos >= 0) {
    /* If the user told us where to position the IAM...*/
    int bytesread = dmkParams->dmk_data_p - (dmkParams->dmk_track + DMK_TKHDR_SIZE);
    if (dmkParams->dmk_awaiting_iam || (unsigned char*) dmkParams->dmk_idam_p == dmkParams->dmk_track) {
      /* First IAM.  (Or a subsequent IAM with no IDAMs in between --
	 in the latter case, we assume the previous IAM(s) were
	 garbage.)  Position the IAM as instructed.  This can result
	 in data loss if an IAM appears somewhere in the middle of the
	 track, unless the read was for twice the track length as the
	 hole=0 (-h0) flag sets it. */
      int iam_pos = dmkParams->dmk_iam_pos;
      if (encoding == FM && !(dmk_header.options & DMK_SDEN_OPT)) {
	iam_pos *= 2;
      }
      if (bytesread < iam_pos) {
	/* Not enough bytes read yet.  Move read bytes forward and add fill. */
	memmove(dmkParams->dmk_track + DMK_TKHDR_SIZE + iam_pos - bytesread,
		dmkParams->dmk_track + DMK_TKHDR_SIZE,
		bytesread);
	memset(dmkParams->dmk_track + DMK_TKHDR_SIZE,
	       (encoding == MFM) ? 0x4e : 0xff,
	       iam_pos - bytesread);
      } else {
	/* Too many bytes read.  Move last iam_pos back and throw rest away. */
	memmove(dmkParams->dmk_track + DMK_TKHDR_SIZE,
		dmkParams->dmk_track + DMK_TKHDR_SIZE + bytesread - iam_pos,
		iam_pos);
      }
      dmkParams->dmk_data_p = dmkParams->dmk_track + DMK_TKHDR_SIZE + iam_pos;
      dmkParams->dmk_awaiting_iam = 0;
    } else {
      /* IAM that follows another IAM and one or more IDAMs.  If we're
	 >95% of the way around the track, assume it's actually the
	 first one again and stop here.  XXX This heuristic might be
	 useful even when the user isn't having us position by IAM. */
      if (bytesread > (dmk_header.tracklen - DMK_TKHDR_SIZE) * 95 / 100) {
	msg(OUT_IDS, "[stopping before second IAM] ");
	dmkParams->dmk_full = 1;
	return;
      }
    }
  }

  dmkParams->dmk_awaiting_dam = 0;
  dmkParams->dmk_valid_id = 0;
  dmk_data(byte, encoding, decoderParams, dmkParams);
}

void dmk_write_header(UserParams_t *userParams, DmkParams_t *dmkParams)
{
  rewind(userParams->dmk_file);
  /* assumes this machine is little-endian: */
  fwrite(&dmk_header, sizeof(dmk_header_t), 1, userParams->dmk_file);
}

void dmk_write(DecoderParams_t *decoderParams, DmkParams_t *dmkParams, UserParams_t *userParams, ReportingValues_t *reportingValues)
{
  int i;

  msg(OUT_TSUMMARY, " %d good sector%s, %d error%s\n",
      decoderParams->good_sectors, plu(decoderParams->good_sectors), decoderParams->errcount, plu(decoderParams->errcount));
  msg(OUT_IDS, "\n");

  reportingValues->total_good_sectors += decoderParams->good_sectors;
  reportingValues->total_errcount += decoderParams->errcount;
  if (decoderParams->errcount) {
    reportingValues->err_tracks++;
  } else if (decoderParams->good_sectors > 0) {
    reportingValues->good_tracks++;
  }
  for (i = 0; i < N_ENCS; i++) {
    total_enc_count[i] += enc_count[i];
  }
  fwrite(dmkParams->dmk_track, dmk_header.tracklen, 1, userParams->dmk_file);
}

void dmk_init_track(DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  int i;

  memset(dmkParams->dmk_track, 0, dmk_header.tracklen);
  dmkParams->dmk_idam_p = (unsigned short*) dmkParams->dmk_track;
  dmkParams->dmk_data_p = dmkParams->dmk_track + DMK_TKHDR_SIZE;
  dmkParams->dmk_awaiting_dam = 0;
  dmkParams->dmk_valid_id = 0;
  dmkParams->dmk_full = 0;
  decoderParams->good_sectors = 0;
  for (i = 0; i < N_ENCS; i++) {
    enc_count[i] = 0;
  }
  decoderParams->errcount = 0;
  decoderParams->backward_am = 0;
  dmkParams->dmk_ignored_count = 0;
  if (dmkParams->dmk_ignore < 0) {
    i = dmkParams->dmk_ignore;
    while (i++) *dmkParams->dmk_data_p++ = 0xff;
  }
  dmkParams->cylseen = -1;
  if (dmkParams->dmk_iam_pos >= 0) {
    dmkParams->dmk_awaiting_iam = 1;
  }
}

void check_missing_dam(DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  if (!dmkParams->dmk_awaiting_dam) return;
  dmkParams->dmk_awaiting_dam = 0;
  dmkParams->dmk_valid_id = 0;
  decoderParams->dbyte = decoderParams->ibyte = -1;
  decoderParams->errcount++;
  msg(OUT_ERRORS, "[missing DAM] ");
}

int dmk_check_wraparound(DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  /* Once we've read 95% of the track, if we see a sector ID that's
     identical to the first one we saw on the track, conclude that we
     wrapped around and are seeing the first one again, and
     retroactively ignore it. */
  unsigned short first_idamp, last_idamp;
  int cmplen;
  if (dmkParams->dmk_data_p - dmkParams->dmk_track - DMK_TKHDR_SIZE <
      (dmk_header.tracklen - DMK_TKHDR_SIZE) * 95 / 100) {
    return 0;
  }
  first_idamp = *(unsigned short*) dmkParams->dmk_track;
  last_idamp = *(dmkParams->dmk_idam_p - 1);
  if (first_idamp == last_idamp) return 0;
  if ((first_idamp & DMK_DDEN_FLAG) != (last_idamp & DMK_DDEN_FLAG)) return 0;
  if ((first_idamp & DMK_DDEN_FLAG) || (dmk_header.options & DMK_SDEN_OPT)) {
    cmplen = 5;
  } else {
    cmplen = 10;
  }
  if (memcmp(&dmkParams->dmk_track[first_idamp & DMK_IDAMP_BITS],
	     &dmkParams->dmk_track[last_idamp & DMK_IDAMP_BITS], cmplen) == 0) {
    msg(OUT_ERRORS, "[wraparound] ");
    *--dmkParams->dmk_idam_p = 0;
    dmkParams->dmk_awaiting_dam = 0;
    decoderParams->ibyte = -1;
    dmkParams->dmk_full = 1;
    return 1;
  }
  return 0;
}

int secsize(int sizecode, int encoding)
{
  switch (encoding) {
  case MFM:
    /* 179x can only do sizes 128, 256, 512, 1024, and ignores
       higher-order bits.  If you need to read a 765-formatted disk
       with larger sectors, change maxsize with the -z
       command line option. */
    return 128 << (sizecode % (maxsize + 1));

  case FM:
  default:
    /* WD1771 has two different encodings for sector size, depending on
       a bit in the read/write command that is not recorded on disk.
       We guess IBM encoding if the size is <= maxsize, non-IBM
       if larger.  This doesn't really matter for demodulating the
       data bytes, only for checking the CRC.  */
    if (sizecode <= maxsize) {
      /* IBM */
      return 128 << sizecode;
    } else {
      /* non-IBM */
      return 16 * (sizecode ? sizecode : 256);
    }

  case RX02:
    return 256 << (sizecode % (maxsize + 1));
  }
}

void init_decoder(DecoderParams_t *decoderParams)
{
  decoderParams->accum   = 0;
  decoderParams->taccum  = 0;
  decoderParams->bits    = 0;
  decoderParams->ibyte   = -1;
  decoderParams->dbyte   = -1;
  decoderParams->premark = 0;
  decoderParams->mark_after = -1;
  decoderParams->curenc  = decoderParams->first_encoding;
}

int mfm_valid_clock(unsigned long long accum)
{
  /* Check for valid clock bits */
  unsigned int xclock = ~((accum >> 1) | (accum << 1)) & 0xaaaa;
  unsigned int clock = accum & 0xaaaa;
  if (xclock != clock) {
    //msg(OUT_ERRORS, "[clock exp %04x got %04x]", xclock, clock);
    return 0;
  }
  return 1;
}

/* Window used to undo RX02 MFM transform */
#if 1
#define WINDOW 4   /* change aligned 1000 -> 0101 */
#else
#define WINDOW 12  /* change aligned x01000100010 -> x00101010100 */
#endif

void change_enc(int newenc, DecoderParams_t *decoderParams)
{
  if (decoderParams->curenc != newenc) {
    msg(OUT_ERRORS, "[%s->%s] ", enc_name[decoderParams->curenc], enc_name[newenc]);
    decoderParams->curenc = newenc;
  }
}

/*
 * Main routine of the FM/MFM/RX02 decoder.  The input is a stream of
 * alternating clock/data bits, passed in one by one.  See decoder.txt
 * for documentation on how the decoder works.
 */
void process_bit(int bit, DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  static int curcyl = 0;
  unsigned char val = 0;
  int i;

  decoderParams->accum = (decoderParams->accum << 1) + bit;
  decoderParams->taccum = (decoderParams->taccum << 1) + bit;
//  accum = (accum << 1) + bit;
//  taccum = (taccum << 1) + bit;
  decoderParams->bits++;
  if (decoderParams->mark_after >= 0) decoderParams->mark_after--;
  if (decoderParams->write_splice > 0) decoderParams->write_splice--;

  /*
   * Pre-detect address marks: we shift bits into the low-order end of
   * our 64-bit shift register (accum), look for marks in the lower
   * mark (or certain other patterns), we repeat or drop some bits to
   * achieve proper clock/data separatation and proper byte-alignment.
   * Pre-detecting the marks lets us do this adjustment earlier and
   * decode data more cleanly.
   *
   * We always sample bits at the MFM rate (twice the FM rate), but
   * we look for both FM and MFM marks at the same time.  There is
   * ambiguity here if we're dealing with normal (not DEC-modified)
   * MFM, because FM marks can be legitimate MFM data.  So we don't
   * look for FM marks while we think we're inside an MFM ID or data
   * block, only in gaps.  With -e2, we don't look for FM marks at
   * all.
   */

  /*
   * For FM and RX02 marks, we look at 9 data bits (including a
   * leading 0), which ends up being 36 bits of accum (2x for clocks,
   * another 2x for the double sampling rate).  We must not look
   * inside a region that can contain standard MFM data.
   */
  if (decoderParams->uencoding != MFM && decoderParams->bits >= 36 && !decoderParams->write_splice &&
      (decoderParams->curenc != MFM || (decoderParams->ibyte == -1 && decoderParams->dbyte == -1 && decoderParams->mark_after == -1))) {
    switch (decoderParams->accum & 0xfffffffffULL) {
    case 0x8aa2a2a88ULL:  /* 0xfc / 0xd7: Index address mark */
    case 0x8aa222aa8ULL:  /* 0xfe / 0xc7: ID address mark */
    case 0x8aa222888ULL:  /* 0xf8 / 0xc7: Standard deleted DAM */
    case 0x8aa22288aULL:  /* 0xf9 / 0xc7: RX02 deleted DAM / WD1771 user DAM */
    case 0x8aa2228a8ULL:  /* 0xfa / 0xc7: WD1771 user DAM */
    case 0x8aa2228aaULL:  /* 0xfb / 0xc7: Standard DAM */
    case 0x8aa222a8aULL:  /* 0xfd / 0xc7: RX02 DAM */
      change_enc(FM, decoderParams);
      if (decoderParams->bits < 64 && decoderParams->bits >= 48) {
	msg(OUT_HEX, "(+%d)", 64 - (decoderParams->bits));
	decoderParams->bits = 64; /* byte-align by repeating some bits */
      } else if (decoderParams->bits < 48 && decoderParams->bits > 32) {
	msg(OUT_HEX, "(-%d)", (decoderParams->bits) - 32);
	decoderParams->bits = 32; /* byte-align by dropping some bits */
      }
      decoderParams->mark_after = 32;
      decoderParams->premark = 0; // doesn't apply to FM marks
      break;

    case 0xa222a8888ULL:  /* Backward 0xf8-0xfd DAM */
      change_enc(FM, decoderParams);
      decoderParams->backward_am++;
      msg(OUT_ERRORS, "[backward AM] ");
      break;
    }
  }

  /*
   * For MFM premarks, we look at 16 data bits (two copies of the
   * premark), which ends up being 32 bits of accum (2x for clocks).
   */
  if (decoderParams->uencoding != FM && decoderParams->uencoding != RX02 &&
      decoderParams->bits >= 32 && !decoderParams->write_splice) {
    switch (decoderParams->accum & 0xffffffff) {
    case 0x52245224:
      /* Pre-index mark, 0xc2c2 with missing clock between bits 3 & 4
	 (using 0-origin big-endian counting!).  Would be 0x52a452a4
	 without missing clock. */
      change_enc(MFM, decoderParams);
      decoderParams->premark = 0xc2;
      if (decoderParams->bits < 64 && decoderParams->bits > 48) {
	msg(OUT_HEX, "(+%d)", 64 - decoderParams->bits);
	decoderParams->bits = 64; /* byte-align by repeating some bits */
      }
      decoderParams->mark_after = decoderParams->bits;
      break;

    case 0x44894489:
      /* Pre-address mark, 0xa1a1 with missing clock between bits 4 & 5
	 (using 0-origin big-endian counting!).  Would be 0x44a944a9
	 without missing clock.  Reading a pre-address mark backward
	 also matches this pattern, but the following byte is then 0x80. */
      change_enc(MFM, decoderParams);
      decoderParams->premark = 0xa1;
      if (decoderParams->bits < 64 && decoderParams->bits > 48) {
	msg(OUT_HEX, "(+%d)", 64 - decoderParams->bits);
	decoderParams->bits = 64; /* byte-align by repeating some bits */
      }
      decoderParams->mark_after = decoderParams->bits;
      break;

    case 0x55555555:
      if (decoderParams->curenc == MFM && decoderParams->mark_after < 0 &&
	  decoderParams->ibyte == -1 && decoderParams->dbyte == -1 && !(decoderParams->bits & 1)) {
	/* ff ff in gap.  This should probably be 00 00, so drop 1/2 bit */
	msg(OUT_HEX, "(-1)");
	decoderParams->bits--;
      }
      break;

    case 0x92549254:
      if (decoderParams->mark_after < 0 && decoderParams->ibyte == -1 && decoderParams->dbyte == -1) {
	/* 4e 4e in gap.  This should probably be byte-aligned */
	change_enc(MFM, decoderParams);
	if (decoderParams->bits < 64 && decoderParams->bits > 48) {
	  /* Byte-align by dropping bits */
	  msg(OUT_HEX, "(-%d)", decoderParams->bits - 48);
	  decoderParams->bits = 48;
	}
      }
      break;
    }
  }

  /* Undo RX02 DEC-modified MFM transform (in decoderParams->taccum) */
#if WINDOW == 4
  if (decoderParams->bits >= WINDOW && (decoderParams->bits & 1) == 0 && (decoderParams->accum & 0xfULL) == 0x8ULL) {
    decoderParams->taccum = (decoderParams->taccum & ~0xfULL) | 0x5ULL;
 }
#else /* WINDOW == 12 */
  if (decoderParams->bits >= WINDOW && (bits & 1) == 0 && (decoderParams->accum & 0x7ffULL) == 0x222ULL) {
    decoderParams->taccum = (decoderParams->taccum & ~0x7ffULL) | 0x154ULL;
  }
#endif

  if (decoderParams->bits < 64) return;

  if (decoderParams->curenc == FM || decoderParams->curenc == MIXED) {
    /* Heuristic to detect being off by some number of bits */
    if (decoderParams->mark_after != 0 && ((decoderParams->accum >> 32) & 0xddddddddULL) != 0x88888888ULL) {
     for (i = 1; i <= 3; i++) {
	if (((decoderParams->accum >> (32 - i)) & 0xddddddddULL) == 0x88888888ULL) {
	  /* Ignore oldest i bits */
	  decoderParams->bits -= i;
	  msg(OUT_HEX, "(-%d)", i);
	  if (decoderParams->bits < 64) return;
	  break;
	}
      }
      if (i > 3) {
#if 0 /* Bad idea: fires way too often in FM gaps. */
	/* Check if it looks more like MFM */
	if (uencoding != FM && uencoding != RX02 &&
	    ibyte == -1 && dbyte == -1 && !write_splice &&
	    (decoderParams->accum & 0xaaaaaaaa00000000ULL) &&
	    (decoderParams->accum & 0x5555555500000000ULL)) {
	  for (i = 1; i <= 2; i++) {
	    if (mfm_valid_clock(accum >> (48 - i))) {
	      change_enc(MFM);
	      decoderParams->bits -= i;
	      msg(OUT_HEX, "(-%d)", i);
	      return;
	    }
	  }
	}	    
#endif
	/* Note bad clock pattern */
	msg(OUT_HEX, "?");
      }
    }
    for (i=0; i<8; i++) {
      val |= (decoderParams->accum & (1ULL << (4*i + 1 + 32))) >> (3*i + 1 + 32);
   }
    decoderParams->bits = 32;

  } else if (decoderParams->curenc == MFM) {
    for (i=0; i<8; i++) {
      val |= (decoderParams->accum & (1ULL << (2*i + 48))) >> (i + 48);
    }
    decoderParams->bits = 48;

  } else /* curenc == RX02 */ {
    for (i=0; i<8; i++) {
      val |= (decoderParams->taccum & (1ULL << (2*i + 48))) >> (i + 48);
    }
    decoderParams->bits = 48;
  }

  if (decoderParams->mark_after == 0) {
    decoderParams->mark_after = -1;
    switch (val) {
    case 0xfc:
      /* Index address mark */
      if (decoderParams->curenc == MFM && decoderParams->premark != 0xc2) break;
      check_missing_dam(decoderParams, dmkParams);
      msg(OUT_IDS, "\n#fc ");
      dmk_iam(0xfc, decoderParams->curenc, decoderParams, dmkParams);
      decoderParams->ibyte = -1;
      decoderParams->dbyte = -1;
      return;

    case 0xfe:
      /* ID address mark */
      if (decoderParams->curenc == MFM && decoderParams->premark != 0xa1) break;
      if (dmkParams->dmk_awaiting_iam) break;
      check_missing_dam(decoderParams, dmkParams);
      msg(OUT_IDS, "\n#fe ");
      dmk_idam(0xfe, decoderParams->curenc, decoderParams, dmkParams);
      decoderParams->crc = calc_crc1((decoderParams->curenc == MFM) ? 0xcdb4 : 0xffff, val);
      decoderParams->dbyte = -1;
      return;

    case 0xf8: /* Standard deleted data address mark */
    case 0xf9: /* WD1771 user or RX02 deleted data address mark */
    case 0xfa: /* WD1771 user data address mark */
    case 0xfb: /* Standard data address mark */
    case 0xfd: /* RX02 data address mark */
      if (dmk_awaiting_track_start(decoderParams, dmkParams) || !dmk_in_range(decoderParams, dmkParams)) break;
      if (decoderParams->curenc == MFM && decoderParams->premark != 0xa1) break;
      if (!dmkParams->dmk_awaiting_dam) {
	msg(OUT_ERRORS, "[unexpected DAM] ");
	decoderParams->errcount++;
	break;
      }
      dmkParams->dmk_awaiting_dam = 0;
      msg(OUT_HEX, "\n");
      msg(OUT_IDS, "#%2x ", val);
      dmk_data(val, decoderParams->curenc, decoderParams, dmkParams);
      if ((decoderParams->uencoding == MIXED || decoderParams->uencoding == RX02) &&
	  (val == 0xfd ||
	   (val == 0xf9 && (total_enc_count[RX02] + enc_count[RX02] > 0 ||
			    decoderParams->uencoding == RX02)))) {
	change_enc(RX02, decoderParams);
      }
      /* For MFM, decoderParams->premark a1a1a1 is included in the CRC */
      decoderParams->crc = calc_crc1((decoderParams->curenc == MFM) ? 0xcdb4 : 0xffff, val);
      decoderParams->ibyte = -1;
      decoderParams->dbyte = secsize(decoderParams->sizecode, decoderParams->curenc) + 2;
      return;

    case 0x80: /* MFM DAM or IDAM premark read backward */
      if (decoderParams->curenc != MFM || decoderParams->premark != 0xc2) break;
      decoderParams->backward_am++;
      msg(OUT_ERRORS, "[backward AM] ");
      break;

    default:
      /* Premark with no mark */
      //msg(OUT_ERRORS, "[dangling premark] ");
      //errcount++;
      break;
    }
  }

  switch (decoderParams->ibyte) {
  default:
    break;
  case 0:
    msg(OUT_IDS, "cyl=");
    curcyl = val;
    break;
  case 1:
    msg(OUT_IDS, "side=");
    break;
  case 2:
    msg(OUT_IDS, "sec=");
    break;
  case 3:
    msg(OUT_IDS, "size=");
    decoderParams->sizecode = val;
    break;
  case 4:
    msg(OUT_HEX, "crc=");
    break;
  case 6:
    if (decoderParams->crc == 0) {
      msg(OUT_IDS, "[good ID CRC] ");
      dmkParams->dmk_valid_id = 1;
    } else {
      msg(OUT_ERRORS, "[bad ID CRC] ");
      decoderParams->errcount++;
      decoderParams->ibyte = -1;
    }
    msg(OUT_HEX, "\n");
    dmkParams->dmk_awaiting_dam = 1;
    dmk_check_wraparound(decoderParams, dmkParams);
    break;
  case 18:
    /* Done with post-ID gap */
    decoderParams->ibyte = -1;
    break;
  }

  if (decoderParams->ibyte == 2) {
    msg(OUT_ERRORS, "%02x ", val);
  } else if (decoderParams->ibyte >= 0 && decoderParams->ibyte <= 3) {
    msg(OUT_IDS, "%02x ", val);
  } else {
    msg(OUT_SAMPLES, "<");
    msg(OUT_HEX, "%02x", val);
    msg(OUT_SAMPLES, ">");
    msg(OUT_HEX, " ", val);
    msg(OUT_RAW, "%c", val);
  }

  dmk_data(val, decoderParams->curenc, decoderParams, dmkParams);

  if (decoderParams->ibyte >= 0) decoderParams->ibyte++;
  if (decoderParams->dbyte > 0) decoderParams->dbyte--;
  decoderParams->crc = calc_crc1(decoderParams->crc, val);

  if (decoderParams->dbyte == 0) {
    if (decoderParams->crc == 0) {
      msg(OUT_IDS, "[good data CRC] ");
      if (dmkParams->dmk_valid_id) {
	if (decoderParams->good_sectors == 0) decoderParams->first_encoding = decoderParams->curenc;
	decoderParams->good_sectors++;
	enc_count[decoderParams->curenc]++;
	dmkParams->cylseen = curcyl;
      }
    } else {
      msg(OUT_ERRORS, "[bad data CRC] ");
      decoderParams->errcount++;
    }
    msg(OUT_HEX, "\n");
    decoderParams->dbyte = -1;
    dmkParams->dmk_valid_id = 0;
    decoderParams->write_splice = WRITE_SPLICE;
    if (decoderParams->curenc == RX02) {
      change_enc(FM, decoderParams);
    }
  }

  /* Predetect bad MFM clock pattern.  Can't detect at decode time
     because we need to look at 17 bits. */
  if (decoderParams->curenc == MFM && decoderParams->bits == 48 && !mfm_valid_clock(decoderParams->accum >> 32)) {
    if (mfm_valid_clock(decoderParams->accum >> 31)) {
      msg(OUT_HEX, "(-1)");
      decoderParams->bits--;
    } else {
      msg(OUT_HEX, "?");
    }
  }
}

/*
 * Convert Catweasel samples to strings of alternating clock/data bits
 * and pass them to process_bit for further decoding.
 * Ad hoc method using two fixed thresholds modified by a postcomp
 * factor.
 */
void process_sample(int sample, DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  static float adj = 0.0;
  int len;

  msg(OUT_SAMPLES, "%d", sample);
  if (decoderParams->uencoding == FM) {
    if (sample + adj <= decoderParams->fmthresh) {
      /* Short */
      len = 2;
    } else {
      /* Long */
      len = 4;
    }
  } else {
    if (sample + adj <= decoderParams->mfmthresh1) {
      /* Short */
      len = 2;
    } else if (sample + adj <= decoderParams->mfmthresh2) {
      /* Medium */
      len = 3;
    } else {
      /* Long */
      len = 4;
    }
    
  }
/*
  if (decoderParams->cwclock <= 4) {
      adj = (sample - (len/2.0 * decoderParams->mfmshort * decoderParams->cwclock)) * decoderParams->postcomp;
  } else {
      adj = (sample - (len/2.0 * decoderParams->mfmshort * (decoderParams->cwclock >> 3))) * decoderParams->postcomp;
  }
*/

  msg(OUT_SAMPLES, "%c %f ", "--sml"[len], adj);

  process_bit(1, decoderParams, dmkParams);
  while (--len) process_bit(0, decoderParams, dmkParams);
}

/* Push out any valid bits left in accum at end of track */
void flush_bits(DecoderParams_t *decoderParams, DmkParams_t *dmkParams)
{
  int i;
  for (i=0; i<63; i++) {
    process_bit(!(i&1), decoderParams, dmkParams);
  }
}

/* Main program */

void cleanup(void)
{
//  catweasel_free_controller(&c);
}

void handler(int sig)
{
  cleanup();
  signal(sig, SIG_DFL);
  kill(getpid(), sig);
}

void set_kind(DecoderParams_t *decoderParams, UserParams_t *userParams)
{
  kind_desc* kd = &kinds[userParams->kind-1];
  decoderParams->cwclock     = kd->cwclock;
  decoderParams->fmthresh    = kd->fmthresh;
  decoderParams->mfmthresh1  = kd->mfmthresh1;
  decoderParams->mfmthresh2  = kd->mfmthresh2;
  decoderParams->dmktracklen = kd->cwtracklen;
  decoderParams->mfmshort    = kd->mfmshort;
}

/* Do a histogram of a track, also counting the total
   number of catweasel clocks to go around the track.
*/
int do_histogram(int drive, int track, int side, int *histogram,
	     int* total_cycles, int* total_samples, float* first_peak, int reverse)
{
//  int b;
  int i, tc, ts;
  float peak;
  int pwidth, psamps, psampsw;

  tc = 0;
  ts = 0;
  for (i=0; i<128; i++) {
    histogram[i] = 0;
  }
//  catweasel_seek(&c.drives[drive], track);
  /*
   * Use index-to-index read without marking index edges.  Although
   * this does count the width of the index pulse twice, it's fast and
   * quite accurate enough for a histogram.
   */
/*
  if (!catweasel_read(&c.drives[drive], side ^ reverse, 1, 0, 0)) {
    return 0;
  }
  while ((b = catweasel_get_byte(&c)) != -1 && b < 0x80) {
    histogram[b & 0x7f]++;
    tc += b + 1;  // not sure if the +1 is right 
    ts++;
  }
*/

  /* Print histogram for debugging */
  for (i=0; i<128; i+=8) {
    msg(OUT_SAMPLES, "%3d: %06d %06d %06d %06d %06d %06d %06d %06d\n",
	i, histogram[i+0], histogram[i+1], histogram[i+2], histogram[i+3],
	histogram[i+4], histogram[i+5], histogram[i+6], histogram[i+7]);
  }

  /* Find first peak */
  i = 0;
  pwidth = 0;
  psamps = 0;
  psampsw = 0;
  while (histogram[i] < 64 && i < 128) i++;
  while (histogram[i] >= 64 && i < 128) {
    pwidth++;
    psamps += histogram[i];
    psampsw += histogram[i] * i;
    i++;
  }
  if (pwidth > 24) {
    /* Track is blank */
    peak = -1.0;
  } else {
    /* again not sure of +1.0 */
    peak = ((float) psampsw) / psamps + 1.0; 
  }

  *total_cycles = tc;
  *total_samples = ts;
  *first_peak = peak;
  return 1;
}

/* Guess the kind of drive and media in use */
void detect_kind(int drive, DecoderParams_t *decoderParams, UserParams_t *userParams)
{
  int hg[128];
  int *histogram = &hg[0];
  int total_cycles, total_samples;
  float peak, rpm, dclock;

  if (!do_histogram(drive, 0, 0, histogram,
		    &total_cycles, &total_samples, &peak, userParams->reverse)) {
    if (decoderParams->hole) {
      fprintf(stderr, "cw2dmk: No index hole detected\n");
    } else {
      fprintf(stderr,
	      "cw2dmk: No index hole; can't detect drive and media type\n");
      fprintf(stderr,
	      "  Try using the -k flag to specify the correct type\n");
    }
    cleanup();
    exit(1);
  }

  if (peak < 0.0) {
    /* Track is blank */
    fprintf(stderr, "cw2dmk: Track 0 side 0 is unformatted\n");
    cleanup();
    exit(1);
  } else {
//    dclock = 7080.5 / peak;
    dclock = (CWHZ / 1000) / peak;
  }

  /* Total cycles gives us the RPM */
 // rpm = 7080500.0 / ((float)total_cycles) * 60.0;
  rpm = CWHZ / ((float)total_cycles) * 60.0;

  msg(OUT_SAMPLES, "Data clock approx %f kHz\n", dclock);
  msg(OUT_SAMPLES, "Drive speed approx %f RPM\n", rpm);

  if (rpm > 270.0 && rpm < 330.0) {
    /* 300 RPM */
    if (dclock > 225.0 && dclock < 287.5) {
      /* Data rate 250 kHz */
      userParams->kind = 2;
    } else if (dclock > 450.0 && dclock < 575.0) {
      /* Data rate 500 kHz */
      userParams->kind = 4;
    }
  } else if (rpm > 330.0 && rpm < 396.0) {
    /* 360 RPM */
    if (dclock > 270.0 && dclock < 345.0) {
      /* Data rate 300 kHz */
      userParams->kind = 1;
    } else if (dclock > 450.0 && dclock < 575.0) {
      /* Data rate 500 kHz */
      userParams->kind = 3;
    }
  }    

  if (userParams->kind == -1) {
    fprintf(stderr, "cw2dmk: Failed to detect drive and media type\n");
    fprintf(stderr, "  Data clock approx %f kHz\n", dclock);
    fprintf(stderr, "  Drive speed approx %f RPM\n", rpm);
    cleanup();
    exit(1);
  }
  set_kind(decoderParams, userParams);

  msg(OUT_QUIET + 1, "Detected %s\n", kinds[userParams->kind-1].description);
  fflush(stdout);
}

/* Check if back side is formatted */
int detect_sides(int drive, int reverse)
{
  int res = 1;
  int hg[128];
  int *histogram = &hg[0];
  int total_cycles, total_samples;
  float peak;

  if (!do_histogram(drive, 0, 1, histogram,
		    &total_cycles, &total_samples, &peak, reverse)) {
    fprintf(stderr,
	    "cw2dmk: No index hole; can't detect if side 1 is formatted\n");
    cleanup();
    exit(1);
  }

  if (peak > 0.0) {
    float dclock = CWHZ / peak * 1000;
    if (dclock > 225.0 && dclock < 575.0) {
      res = 2;
    }
  }
  msg(OUT_QUIET + 1, "Detected side 1 %sformatted\n",
      res == 1 ? "not " : "");
  fflush(stdout);
  return res;
}

void usage(DecoderParams_t *decoderParams, UserParams_t *userParams, DmkParams_t *dmkParams)
{
  printf("\nUsage: cw2dmk_4.4_mod [options] file.dmk\n");
  printf("\n Options [defaults in brackets]:\n");
  printf(" -d drive      Drive unit, 0 or 1, or -1 to autodetect [%d]\n",
	 userParams->drive);
  printf(" -v verbosity  Amount of output [%d]\n", out_level);
  printf("               0 = No output\n");
  printf("               1 = Summary of disk\n");
  printf("               2 = + summary of each track\n");
  printf("               3 = + individual errors\n");
  printf("               4 = + track IDs and DAMs\n");
  printf("               5 = + hex data and event flags\n");
  printf("               6 = like 4, but with raw data too\n");
  printf("               7 = like 5, but with Catweasel samples too\n");
  printf("               21 = level 2 to logfile, 1 to screen, etc.\n");
  printf(" -u logfile    Log output to the given file [none]\n");
  printf(" -j dumpfile   Reads Catweasel data from the given file\n");
  printf("\n Options to manually set values that are normally autodetected\n");
  printf(" -p port       I/O port base (MK1) or card number (MK3/4) [%d]\n",
	 userParams->port);
  printf(" -k kind       1 = %s\n", kinds[0].description);
  printf("               2 = %s\n", kinds[1].description);
  printf("               3 = %s\n", kinds[2].description);
  printf("               4 = %s\n", kinds[3].description);
  printf(" -m steps      Step multiplier, 1 or 2\n");
  printf(" -t tracks     Number of tracks per side\n");
  printf(" -s sides      Number of sides\n");
  printf(" -e encoding   1 = FM (SD), 2 = MFM (DD or HD), 3 = RX02\n");
  printf(" -w fmtimes    Write FM bytes 1 or 2 times [%d]\n", userParams->fmtimes);
  printf("\n Special options for hard to read diskettes\n");
  printf(" -x retries    Number of retries on errors [%d]\n", userParams->retries);
  printf(" -a alternate  Alternate even/odd tracks on retries with -m2 [%d]\n",
	 userParams->alternate);
  printf("               0 = always even\n");
  printf("               1 = always odd\n");
  printf("               2 = even, then odd\n");
  printf("               3 = odd, then even\n");
  printf(" -o postcomp   Amount of read-postcompensation (0.0-1.0) [%.2f]\n",
	 decoderParams->postcomp);
  printf(" -h hole       Track start: 1 = index hole, 0 = anywhere [%d]\n",
	 decoderParams->hole);
  printf(" -g ign        Ignore first ign bytes of track [%d]\n", dmkParams->dmk_ignore);
  printf(" -i ipos       Force IAM to ipos from track start; "
	 "if -1, don't [%d]\n", dmkParams->dmk_iam_pos);
  printf(" -z maxsize    Allow sector sizes up to 128<<maxsize [%d]\n",
	 maxsize);
  printf(" -r reverse    0 = normal, 1 = reverse sides [%d]\n", userParams->reverse);
  printf("\n Fine-tuning options; effective only after the -k option\n");
  printf(" -c clock      Catweasel clock multipler [%d]\n", decoderParams->cwclock);
  printf(" -1 threshold  MFM threshold for short vs. medium [%d]\n",
	 decoderParams->mfmthresh1);
  printf(" -2 threshold  MFM threshold for medium vs. long [%d]\n",
	 decoderParams->mfmthresh2);
  printf(" -f threshold  FM-only (-e1) threshold for short vs. long [%d]\n",
	 decoderParams->fmthresh);
  printf(" -l bytes      DMK track length in bytes [%d]\n", decoderParams->dmktracklen);
  printf("\n");
  exit(1);
}

int main(int argc, char** argv)
{
  int ch, currentTrack, side, headpos, i;//, readtime;
  int guess_sides = 0, guess_steps = 0, guess_tracks = 0;
  
  DecoderParams_t dps, *decoderParams = &dps;
  initializeDecoderParams(decoderParams);
  
  UserParams_t ups, *userParams = &ups;
  initializeUserParams(userParams);

  DmkParams_t dmkps, *dmkParams = &dmkps;
  dmk_header_t dhr;
  initializeDmkParams(dmkParams, &dhr);
  
  ReportingValues_t rvs, *reportingValues = &rvs;
  initializeReportingValues(reportingValues);
  
  // main
  int prevcylseen;
  int flippy = 0;
  
  CWImageHeader_t cwih, *cwImageHeader = &cwih;
  cwImageHeader->signature[0] = 'C';
  cwImageHeader->signature[1] = 'W';
  cwImageHeader->signature[2] = 'I';
  
  CWImageTrackHeader_t cwith, *cwImageTrackHeader = &cwith;
  
 
    long headerPointer = 0;
    long dataPointer = 0;
    int trackSize;
#define HISTOGRAMSIZE 256
   int histogram[HISTOGRAMSIZE];
  
  printf("Argc: %d\n", argc);
  opterr = 0;
  for (;;) {
    ch = getopt(argc, argv, "p:d:v:u:k:m:t:s:e:w:x:a:o:h:g:i:z:r:c:1:2:f:l:j:"); // left: b, n, q, y
    if (ch == -1) break;
    switch (ch) {
    case 'p':
      userParams->port = strtol(optarg, NULL, 16);
      if (userParams->port < 0 || (userParams->port >= MK3_MAX_CARDS && userParams->port < MK1_MIN_PORT) ||
	  (userParams->port > MK1_MAX_PORT)) {
	fprintf(stderr,
		"cw2dmk: -p must be between 0x%x and 0x%x for MK3/4 cards,\n"
		"  or between 0x%x and 0x%x for MK1 cards.\n",
		0, MK3_MAX_CARDS-1, MK1_MIN_PORT, MK1_MAX_PORT);
	exit(1);
      }
      break;
    case 'd':
      userParams->drive = strtol(optarg, NULL, 0);
      if (userParams->drive < -1 || userParams->drive > 1) usage(decoderParams, userParams, dmkParams);
      break;
    case 'v':
      out_level = strtol(optarg, NULL, 0);
      if (out_level < OUT_QUIET || out_level > OUT_SAMPLES * 11) {
	usage(decoderParams, userParams, dmkParams);
      }
      out_file_level = out_level / 10;
      out_level = out_level % 10;
      break;
    case 'u':
      out_file_name = optarg;
      break;
    case 'k':
      userParams->kind = strtol(optarg, NULL, 0);
      if (userParams->kind < 1 || userParams->kind > NKINDS) usage(decoderParams, userParams, dmkParams);
      set_kind(decoderParams ,userParams);
      break;
    case 'm':
      userParams->steps = strtol(optarg, NULL, 0);
      if (userParams->steps < 1 || userParams->steps > 2) usage(decoderParams, userParams, dmkParams);
      break;
    case 't':
      userParams->tracks = strtol(optarg, NULL, 0);
      if (userParams->tracks < 0 || userParams->tracks > 85) usage(decoderParams, userParams, dmkParams);
      break;
    case 's':
      userParams->sides = strtol(optarg, NULL, 0);
      if (userParams->sides < 1 || userParams->sides > 2) usage(decoderParams, userParams, dmkParams);
      break;
    case 'e':
      decoderParams->uencoding = strtol(optarg, NULL, 0);
      decoderParams->uencoding = strtol(optarg, NULL, 0);
      if (decoderParams->uencoding < FM || decoderParams->uencoding > RX02) usage(decoderParams, userParams, dmkParams);
      break;
    case 'w':
      userParams->fmtimes = strtol(optarg, NULL, 0);
      if (userParams->fmtimes != 1 && userParams->fmtimes != 2) usage(decoderParams, userParams, dmkParams);
      break;
    case 'x':
      userParams->retries = strtol(optarg, NULL, 0);
      if (userParams->retries < 0) usage(decoderParams, userParams, dmkParams);
      break;
    case 'a':
      userParams->alternate = strtol(optarg, NULL, 0);
      if (userParams->alternate < 0 || userParams->alternate > 3) usage(decoderParams, userParams, dmkParams);
      break;
    case 'o':
      decoderParams->postcomp = strtod(optarg, NULL);
      if (decoderParams->postcomp < 0.0 || decoderParams->postcomp > 1.0) usage(decoderParams, userParams, dmkParams);
      break;
    case 'h':
      decoderParams->hole = strtol(optarg, NULL, 0);
      if (decoderParams->hole < 0 || decoderParams->hole > 1) usage(decoderParams, userParams, dmkParams);
      break;
    case 'g':
      dmkParams->dmk_ignore = strtol(optarg, NULL, 0);
      break;
    case 'i':
      dmkParams->dmk_iam_pos = strtol(optarg, NULL, 0);
      break;
    case 'z':
      maxsize = strtol(optarg, NULL, 0);
      if (maxsize < 0 || maxsize > 255) usage(decoderParams, userParams, dmkParams);
      break;
    case 'r':
      userParams->reverse = strtol(optarg, NULL, 0);
      if (userParams->reverse < 0 || userParams->reverse > 1) usage(decoderParams, userParams, dmkParams);
      break;
    case 'c':
      decoderParams->cwclock = strtol(optarg, NULL, 0);
      if (userParams->kind == -1 || (decoderParams->cwclock != 1 && decoderParams->cwclock != 2 && decoderParams->cwclock != 4 &&
              decoderParams->cwclock != 8 && decoderParams->cwclock != 16 && decoderParams->cwclock != 32)) {
	usage(decoderParams, userParams, dmkParams);
      }
      break;
    case '1':
      decoderParams->mfmthresh1 = strtol(optarg, NULL, 0);
      if (userParams->kind == -1) usage(decoderParams, userParams, dmkParams);
      break;
    case '2':
      decoderParams->mfmthresh2 = strtol(optarg, NULL, 0);
      if (userParams->kind == -1) usage(decoderParams, userParams, dmkParams);
      break;
    case 'f':
      decoderParams->fmthresh = strtol(optarg, NULL, 0);
      if (userParams->kind == -1) usage(decoderParams, userParams, dmkParams);
      break;
    case 'l':
      decoderParams->dmktracklen = strtol(optarg, NULL, 0);
      if (decoderParams->dmktracklen < 0 || decoderParams->dmktracklen > 0x4000) usage(decoderParams, userParams, dmkParams);
      if (userParams->kind == -1) usage(decoderParams, userParams, dmkParams);
      break;
    case 'b':
      userParams->dumpFileName = optarg;
      if (userParams->dumpFileName == NULL) usage(decoderParams, userParams, dmkParams);
      break; 
    case 'j':
      userParams->dumpFileName = optarg;
      if (userParams->dumpFileName == NULL) usage(decoderParams, userParams, dmkParams);
      userParams->readFromFile = 1;
      break; 
    default:
      usage(decoderParams, userParams, dmkParams);
      break;
    }
  }

  if (optind >= argc) {
    usage(decoderParams, userParams, dmkParams);
  }
  
  if (out_file_name && out_file_level == OUT_QUIET) {
    /* Default: log to file at same level as screen */
    out_file_level = out_level;
  }
  if (!out_file_name && out_file_level > OUT_QUIET) {
    char *p;
    int len;

    p = strrchr(argv[optind], '.');
    if (p == NULL) {
      len = strlen(argv[optind]);
    } else {
      len = p - argv[optind];
    }
    out_file_name = (char *) malloc(len + 5);
    sprintf(out_file_name, "%.*s.log", len, argv[optind]);
  }

  /* Keep drive from spinning endlessly on (expected) signals */
  signal(SIGHUP, handler);
  signal(SIGINT, handler);
  signal(SIGQUIT, handler);
  signal(SIGPIPE, handler);
  signal(SIGTERM, handler);


  /* Open log file if needed */
  if (out_file_name) {
    out_file = fopen(out_file_name, "w");
    if (out_file == NULL) {
      perror(out_file_name);
      exit(1);
    }
  }

  // Open dump file if needed */
  if (userParams->dumpFileName != NULL) {
      if (userParams->readFromFile) {
          userParams->dump_file = fopen(userParams->dumpFileName, "rb");
          if (userParams->dump_file == NULL) {
              printf("Error opening dump file '%s' for reading\n", userParams->dumpFileName);
              perror(userParams->dumpFileName);
              exit(1);
          }
          readFromImage(cwImageHeader, 0, sizeof(CWImageHeader_t), userParams->dump_file);
          printf("Signature: %c%c%c\n", cwImageHeader->signature[0], cwImageHeader->signature[1], cwImageHeader->signature[2]);

          if (memcmp(&cwiSig, cwImageHeader->signature, sizeof(cwiSig) - 1) == 0) {
              printf("CWI signature found, clock = %d\n", cwImageHeader->clock - '0');
              decoderParams->cwclock = cwImageHeader->clock - '0';
          } else if (memcmp(&dfiSig, cwImageHeader->signature, sizeof(dfiSig) - 1) == 0) {
              printf("DFI signature found, clock = %d\n", decoderParams->cwclock);
          } else {
              printf("Not a CWI or DFE file (%d)\n", sizeof(dfiSig));
              exit(1);
          }
          
          
      } else {
          userParams->dump_file = fopen(userParams->dumpFileName, "wb");
          if (userParams->dump_file == NULL) {
              printf("Error opening dump file '%s' for writing\n", userParams->dumpFileName);
              perror(userParams->dumpFileName);
              exit(1);
          }
        cwImageHeader->clock = decoderParams->cwclock + '0';
        fwrite(cwImageHeader, sizeof(CWImageHeader_t), 1, userParams->dump_file);
      } 
  }
  

  /* Log the version number and command line */
  msg(OUT_TSUMMARY, "cw2dmk %s\n", VERSION);
  msg(OUT_ERRORS, "Command line: ");
  for (i = 0; i < argc; i++) {
    msg(OUT_ERRORS, "%s ", argv[i]);
  }
  msg(OUT_ERRORS, "\n");

  /* Finish detecting and initializating Catweasel */
/*
  if (userParams->port == -1) {
    userParams->port = MK1_DEFAULT_PORT;
    msg(OUT_SUMMARY, "Failed to detect Catweasel MK3/4 on PCI bus; "
	"looking for MK1 on ISA bus at 0x%x\n", userParams->port);
    fflush(stdout);
  }
*/
/*
  ch = catweasel_init_controller(&c, userParams->port, cw_mk, getenv("CW4FIRMWARE"))
    && catweasel_memtest(&c);

 
  if (ch) {
    msg(OUT_SUMMARY, "Detected Catweasel MK%d at port 0x%x\n", cw_mk, userParams->port);
    fflush(stdout);
  } else {
    fprintf(stderr, "cw2dmk: Failed to detect Catweasel at port 0x%x\n", userParams->port);
    exit(1);
  }
  if (cw_mk == 1 && decoderParams->cwclock == 4) {
    fprintf(stderr, "cw2dmk: Catweasel MK1 does not support 4x clock\n");
    exit(1);
  }
*/

  /* Detect drive */
/*
  if (userParams->drive == -1) {
    for (userParams->drive = 0; userParams->drive < 2; userParams->drive++) {
      msg(OUT_SUMMARY, "Looking for drive %d...", userParams->drive);
      fflush(stdout);
      catweasel_detect_drive(&c.drives[userParams->drive]);
      if (c.drives[userParams->drive].type == 1) {
	msg(OUT_SUMMARY, "detected\n");
	break;
      } else {
	msg(OUT_SUMMARY, "not detected\n");
      }
    }
    if (userParams->drive == 2) {
      fprintf(stderr, "cw2dmk: Failed to detect any drives\n");
      cleanup();
      exit(1);
    }
  } else {
    msg(OUT_SUMMARY, "Looking for drive %d...", userParams->drive);
    fflush(stdout);
    catweasel_detect_drive(&c.drives[userParams->drive]);
    if (c.drives[userParams->drive].type == 1) {
      msg(OUT_SUMMARY, "detected\n");
    } else {
      msg(OUT_SUMMARY, "not detected\n");
      fprintf(stderr, "cw2dmk: Drive %d not detected; proceeding anyway\n",
	      userParams->drive);
    }
  }
*/
  
  userParams->dmkFileName = argv[optind];
  /* Open output file */
  userParams->dmk_file = fopen(userParams->dmkFileName, "wb");
  if (userParams->dmk_file == NULL) {
    perror(userParams->dmkFileName);
    cleanup();
    exit(1);
  }

/*
   Select drive, start motor, wait for spinup 
  catweasel_select(&c, !userParams->drive, userParams->drive);
  catweasel_set_motor(&c.drives[userParams->drive], 1);
  catweasel_usleep(500000);
*/

  /* Guess various parameters if not supplied */
/*
  if (userParams->kind == -1) detect_kind(userParams->drive, decoderParams, userParams);
  if (userParams->sides == -1) {
    userParams->sides = detect_sides(userParams->drive, userParams->reverse);
    guess_sides = 1; / * still allow this guess to be changed * /
  }
  if (userParams->steps == -1) {
    if (userParams->kind == 1) {
      userParams->steps = 2;
    } else {
      userParams->steps = 1;
    }
    guess_steps = 1;
  }
  if (userParams->tracks == -1) {
    userParams->tracks = TRACKS_GUESS / userParams->steps;
    guess_tracks = 1;
  }
*/

      /*
       * With CW MK3, hardware hole-to-hole read can't be made to
       * store the hole locations in the data stream.  Use timed read
       * instead.  Read an extra 10% in case of sectors wrapping past
       * the hole.
       */
//      readtime = 1.1 * kinds[userParams->kind-1].readtime;    


 restart:
  if (guess_sides || guess_steps || guess_tracks) {
    msg(OUT_SUMMARY,
	"Trying %d side%s, %d tracks/side, %s stepping, %s encoding\n",
	userParams->sides, plu(userParams->sides), userParams->tracks, (userParams->steps == 1) ? "single" : "double",
	enc_name[decoderParams->uencoding]);
  }
  fflush(stdout);
  reportingValues->total_errcount = 0;
  reportingValues->total_retries = 0;
  reportingValues->total_good_sectors = 0;
  for (i = 0; i < N_ENCS; i++) {
    total_enc_count[i] = 0;
  }
  reportingValues->good_tracks = 0;
  reportingValues->err_tracks = 0;
  decoderParams->first_encoding = (decoderParams->uencoding == RX02 ? FM : decoderParams->uencoding);

  /* Set DMK parameters */
  memset(&dmk_header, 0, sizeof(dmk_header_t));
  dmk_header.ntracks = userParams->tracks;
  dmk_header.tracklen = decoderParams->dmktracklen; 
  dmk_header.options = ((userParams->sides == 1) ? DMK_SSIDE_OPT : 0) +
                       ((userParams->fmtimes == 1) ? DMK_SDEN_OPT : 0) +
                       ((decoderParams->uencoding == RX02) ? DMK_RX02_OPT : 0);
  dmk_write_header(userParams, dmkParams);
  if (dmkParams->dmk_track) free(dmkParams->dmk_track);
  dmkParams->dmk_track = (unsigned char*) malloc(decoderParams->dmktracklen);
  
  headerPointer = sizeof(CWImageHeader_t);
  dataPointer   = sizeof(CWImageHeader_t) + sizeof(CWImageTrackHeader_t);
  
    int result;
    #define BUFSIZE CWMAXTRACKSIZE
    unsigned char bf[BUFSIZE], *buffer = &bf[0];
  while(1) {
    result = readFromImage(cwImageTrackHeader, headerPointer, sizeof(CWImageTrackHeader_t), userParams->dump_file);
    if (result == 0) break;

    currentTrack = (cwImageTrackHeader->trackMSB << 8) + cwImageTrackHeader->trackLSB;
    side  = (cwImageTrackHeader->headMSB << 8)  + cwImageTrackHeader->headLSB;
    trackSize = trackSizeComposer(cwImageTrackHeader);

    msg(OUT_TSUMMARY, "Header: Track: %d, Side: %d, Size: %02X\n", currentTrack, side, trackSize);
    if (userParams->sides == 1 && side > 0) {
        headerPointer += trackSizeComposer(cwImageTrackHeader) + sizeof(CWImageTrackHeader_t);
        dataPointer   += trackSizeComposer(cwImageTrackHeader) + sizeof(CWImageTrackHeader_t);
        continue;
    }
    result = readFromImage(buffer, dataPointer, trackSize, userParams->dump_file);
   
  /* Loop over tracks */
//  for (currentTrack=0; currentTrack < userParams->tracks; currentTrack++) {
    prevcylseen = dmkParams->cylseen;
    headpos = currentTrack * userParams->steps + ((userParams->steps == 2) ? (userParams->alternate & 1) : 0);

    /* Loop over sides */
//    for (side=0; side < userParams->sides; side++) {
      int retry = 0;

      /* Loop over retries */
//      do {
	int b = 0, completeSample = 0;
//#if DEBUG3
        if (out_level >= OUT_RAW) {
            int i;
            for (i=0; i<HISTOGRAMSIZE; i++) histogram[i] = 0;
        }
//#endif
        
	if (retry) {
	  msg(OUT_TSUMMARY, "[%d good, %d error%s; retry %d] ",
	      decoderParams->good_sectors, decoderParams->errcount, plu(decoderParams->errcount), retry);
	} else {
	  msg(OUT_TSUMMARY, "Track %d, side %d: ", currentTrack, side);
	}
	fflush(stdout);

	/* Seek to correct track */
	if ((userParams->steps == 2) && (retry > 0) && (userParams->alternate & 2)) {
	  headpos ^= 1;
	}
//	catweasel_seek(&c.drives[userParams->drive], headpos);
	dmk_init_track(decoderParams, dmkParams);
        decoderParams->write_splice = 0;
	init_decoder(decoderParams);

	/* Loop over samples */
	decoderParams->index_edge = 0;
         for (i = 0; i < trackSize; i++) {
             /* Detect but ignore the index hole */
             b = buffer[i];
             if (b > 0x7F) {
                 msg(OUT_RAW, "i");
                 b = b & 0x7F;
             }
             if (b == 0x7f) { 
                 completeSample += b;
                 msg(OUT_RAW, "c");
                 continue;
             } else {
                 completeSample += b;
                 msg(OUT_RAW, "s%02X ",completeSample);
             }
          

	  if (out_level >= OUT_RAW) histogram[(completeSample < HISTOGRAMSIZE -1  ? completeSample : HISTOGRAMSIZE - 1)]++;
          

	  /* Process this sample */
//	  process_sample(b, decoderParams, dmkParams);
	  process_sample(completeSample, decoderParams, dmkParams);
          completeSample = 0;
	}
        headerPointer += trackSizeComposer(cwImageTrackHeader) + sizeof(CWImageTrackHeader_t);
        dataPointer   += trackSizeComposer(cwImageTrackHeader) + sizeof(CWImageTrackHeader_t);

        if (out_level >= OUT_RAW) {
            msg(OUT_RAW, "\n");
            /* Print histogram for debugging */
            for (i=0; i<HISTOGRAMSIZE; i+=8) {
              msg(OUT_RAW, "%3d: %06d %06d %06d %06d %06d %06d %06d %06d\n", i,
                     histogram[i+0], histogram[i+1], histogram[i+2],
                     histogram[i+3], histogram[i+4], histogram[i+5],
                     histogram[i+6], histogram[i+7]);
            }
        }


        /* write the header and track buffer to a file */
        flush_bits(decoderParams, dmkParams);
	check_missing_dam(decoderParams, dmkParams);
	if (decoderParams->ibyte != -1) {
	  /* Ignore incomplete sector IDs; assume they are wraparound */
	  msg(OUT_IDS, "[wraparound] ");
	  *--dmkParams->dmk_idam_p = 0;
	}
	if (decoderParams->dbyte != -1) {
	  decoderParams->errcount++;
	  msg(OUT_ERRORS, "[incomplete sector data] ");
	}
	msg(OUT_IDS, "\n");
	if (currentTrack == 0 && side == 1 && decoderParams->good_sectors == 0 && 
	    decoderParams->backward_am >= 9 && decoderParams->backward_am > decoderParams->errcount) {
	  msg(OUT_ERRORS, "[possibly a flippy disk] ");
	  flippy = 1;
	}
	if (guess_sides && side == 1) {
	  guess_sides = 0;
	  if (decoderParams->good_sectors == 0) {
	    msg(OUT_QUIET + 1, "[apparently single-sided; restarting]\n");
	    userParams->sides = 1;
	    goto restart;
	  }
	}
	if (guess_steps) {
	  if (currentTrack == 3) guess_steps = 0;
	  if (userParams->steps == 1) {
	    if ((currentTrack & 1) &&
		(decoderParams->good_sectors == 0 ||
		 dmkParams->cylseen == currentTrack - 1 || dmkParams->cylseen == currentTrack + 1)) {
	      msg(OUT_QUIET + 1,
		  "[double-stepping apparently needed; restarting]\n");
	      userParams->steps = 2;
	      if (guess_tracks) userParams->tracks = TRACKS_GUESS / userParams->steps;
	      goto restart;
	    }
	  } else {
	    if (decoderParams->good_sectors && currentTrack > 0 && dmkParams->cylseen == currentTrack * 2) {
	      msg(OUT_QUIET + 1,
		"[single-stepping apparently needed; restarting]\n");
	      userParams->steps = 1;
	      if (guess_tracks) userParams->tracks = TRACKS_GUESS / userParams->steps;
	      goto restart;
	    }	      
	  }
	}
	if (guess_tracks && (currentTrack == 35 || currentTrack >= 40) &&
	    (decoderParams->good_sectors == 0 ||
	     (side == 0 && dmkParams->cylseen == prevcylseen) ||
	     (side == 0 && currentTrack >= 80 && dmkParams->cylseen == currentTrack/2))) {
	  msg(OUT_QUIET + 1, "[apparently only %d tracks; done]\n", currentTrack);
	  dmk_header.ntracks = currentTrack;
	  dmk_write_header(userParams, dmkParams);
	  goto done;
	}
//      } while (++retry <= userParams->retries && decoderParams->errcount > 0);
      
      reportingValues->total_retries += retry;
      fflush(stdout);
      dmk_write(decoderParams, dmkParams, userParams, reportingValues);
//    }
  }
 done:
    /* Optionally close the dump file */
    if (userParams->dump_file) {
          fclose(userParams->dump_file);
      }            
 
  cleanup();
  if (total_enc_count[RX02] > 0 && decoderParams->uencoding != RX02) {
    // XXX What if disk had some 0xf9 DAM sectors misinterpreted as
    // WD1771 FM instead of RX02-MFM before we detected RX02?  Ugh.
    // Should at least detect this and give an error.  Maybe
    // autorestart if it happens.  I believe it's quite unlikely, as I
    // suspect the 0xf9 DAM is never or almost never actually used on
    // RX02 disks.
    dmk_header.options |= DMK_RX02_OPT;
    dmk_write_header(userParams, dmkParams);
  }
 
 
  msg(OUT_SUMMARY, "\nTotals:\n");
  msg(OUT_SUMMARY,
      "%d good track%s, %d good sector%s (%d FM + %d MFM + %d RX02)\n",
      reportingValues->good_tracks, plu(reportingValues->good_tracks),
      reportingValues->total_good_sectors, plu(reportingValues->total_good_sectors),
      total_enc_count[FM], total_enc_count[MFM], total_enc_count[RX02]);
  msg(OUT_SUMMARY, "%d bad track%s, %d unrecovered error%s, %d retr%s\n",
      reportingValues->err_tracks, plu(reportingValues->err_tracks), reportingValues->total_errcount, plu(reportingValues->total_errcount),
      reportingValues->total_retries, (reportingValues->total_retries == 1) ? "y" : "ies");
  if (flippy) {
    msg(OUT_SUMMARY, "Possibly a flippy disk; check reverse side too\n");
  }
  return 0;
}

void initializeCWTrackHeader(CWImageTrackHeader_t *cwImageTrackHeader, int track, int head, int sector, int trackSize) {
    cwImageTrackHeader->trackMSB  = track >> 8;
    cwImageTrackHeader->trackLSB  = track & 0xFF;
    cwImageTrackHeader->headMSB   = head >>8;
    cwImageTrackHeader->headLSB   = head & 0xFF;
    cwImageTrackHeader->sectorMSB = sector >> 8;
    cwImageTrackHeader->sectorLSB = sector & 0xFF;
    cwImageTrackHeader->trackSizeMMSB = trackSize >> 24;
    cwImageTrackHeader->trackSizeMSB  = (trackSize >> 16) & 0xFF;
    cwImageTrackHeader->trackSizeLSB  = (trackSize >> 8) &0xFF;
    cwImageTrackHeader->trackSizeLLSB = trackSize & 0xFF;
}

void initializeDecoderParams(DecoderParams_t *decoderParams) {
    decoderParams->accum  = 0;
    decoderParams->taccum = 0;
    decoderParams->bits   = 0;
    decoderParams->ibyte  = 0;
    decoderParams->dbyte  = 0;
    decoderParams->premark = 0;
    decoderParams->mark_after = -1;
    decoderParams->curenc = 0;
    decoderParams->crc    = 0;
    decoderParams->sizecode = 0;
    decoderParams->first_encoding = 0;
    decoderParams->write_splice = 0;
    decoderParams->cwclock     = -1;
    decoderParams->fmthresh    = -1;
    decoderParams->mfmthresh1  = -1;
    decoderParams->mfmthresh2  = -1;
    decoderParams->dmktracklen = -1;
    decoderParams->uencoding = MIXED;
    decoderParams->backward_am  = 0;
    decoderParams->good_sectors = 0;
    decoderParams->errcount = 0;  
    decoderParams->mfmshort = -1.0;
    decoderParams->postcomp = 0.5;  
    decoderParams->index_edge = 0;
    decoderParams->hole    = 1;
    decoderParams->maxsize = 3;
}

void initializeUserParams(UserParams_t *userParams) {
    userParams->port = 0;
    userParams->tracks = -1;
    userParams->sides  = -1;
    userParams->steps  = -1;
    userParams->drive  = -1;
    userParams->retries = 4;
    userParams->alternate = 0;
    userParams->fmtimes = 2; /* record FM bytes twice; see man page */
    userParams->kind = -1;
    userParams->reverse = 0;
    userParams->dmk_file    = NULL;
    userParams->dmkFileName = NULL;
    userParams->dump_file    = NULL;
    userParams->dumpFileName = NULL; 
    userParams->readFromFile = 0;
}

void initializeDmkParams(DmkParams_t *dmkParams, dmk_header_t *dmk_header) {
    dmkParams->dmk_idam_p = NULL;
    dmkParams->dmk_data_p = NULL;
    dmkParams->dmk_track  = NULL;
//    dmkParams->dmk_header = dmk_header;
    dmkParams->dmk_valid_id     = 0;
    dmkParams->dmk_awaiting_dam = 0;
    dmkParams->dmk_awaiting_iam = 0;
    dmkParams->dmk_iam_pos      = -1;
    dmkParams->dmk_ignore        = 1; // 0 makes it ignore first track
    dmkParams->dmk_ignored_count = 0;
    dmkParams->dmk_full = 0;
    dmkParams->cylseen  = -1;
}

void initializeReportingValues(ReportingValues_t *reportingValues) {
    reportingValues->total_errcount     = 0;
    reportingValues->total_good_sectors = 0;
    reportingValues->good_tracks = 0;
    reportingValues->err_tracks  = 0;  
    reportingValues->total_retries = 0;
}

int readFromImage(void *data, long location, int size, FILE *fileHandle) {
    int result;
    printf("Reading %05X bytes at location %05lX\n", size, location);
    
    result = fseek(fileHandle, location, SEEK_SET);
    if (result != 0) {
        printf("Error during fseek on file: got %d\n", result);
        return(0);        
    }
    
    result = fread(data, sizeof(char), size, fileHandle);    
    if (result != size) {
        printf("Error reading bytes file: got %d, not %d\n", result, size);
        return(0);
    }
    
//    unsigned char *myData = data;
    
    
    return 1;
}

int trackSizeComposer(CWImageTrackHeader_t *cwImageTrackHeader) {
    int trackSize =  (cwImageTrackHeader->trackSizeMMSB << 24) + (cwImageTrackHeader->trackSizeMSB << 16) +
                     (cwImageTrackHeader->trackSizeLSB  <<  8) +  cwImageTrackHeader->trackSizeLLSB;
//    printf("Track size: %04X\n", trackSize);
    return trackSize;
}

